/**
* \file: ITouchFacade.h
*
* \version: 0.3
*
* \release: $Name:$
*
* Generic interface for touch facade and associated callbacks to be implemented
* by the ISPI
*
* \component: Unified SPI
*
* \author: P. Acar / ADIT/SW2 / pacar@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef USPI_I_TOUCH_FACADE_H
#define USPI_I_TOUCH_FACADE_H

#include <string>

namespace adit { namespace uspi {

// todo carry to some common place for all uspi backends
/** @enum   UspiLogLevel
 *  @brief  Enumeration holding the log levels
 *  @var    USPI_LOG_FATAL: fatal error
 *  @var    USPI_LOG_ERROR: error
 *  @var    USPI_LOG_WARN: warning
 *  @var    USPI_LOG_INFO: informational
 *  @var    USPI_LOG_DEBUG: debug
 *  @var    USPI_LOG_VERBOSE: extensive debug
 */
enum UspiLogLevel
{
    USPI_LOG_FATAL,
    USPI_LOG_ERROR,
    USPI_LOG_WARN,
    USPI_LOG_INFO,
    USPI_LOG_DEBUG,
    USPI_LOG_VERBOSE
};

/** @struct TouchEvent
 *  @brief  Structure holding the touch event data
 *  @var    xRelative: x value relative to the display width, range: 0-1
 *  @var    yRelative: y value relative to the display height, range: 0-1
 *  @var    fingerId: ID of the touching finger
 *  @var    pressure: pressure of the touch, range: 0-1
 *  @var    eventType: touch down, finger up, or motion
 */
struct TouchEvent
{
    float xRelative, yRelative;
    int fingerId;
    float pressure;
    enum touchEventType{ down, move, up } eventType;
};

/** @interface  ITouchFacadeCallbacks
 *  @brief      Callbacks to notify the SPI component. This interface must be
 *              implemented by the SPI component using USPI touch.
 */
class ITouchFacadeCallbacks
{
public:
    virtual ~ITouchFacadeCallbacks() { };

    /** @brief callback initiated with each touch event
     *  @param inEvent: data related to current touch event
     */
    virtual void onTouch(TouchEvent inEvent) = 0;

    /** @brief callback initiated when an error happens
     *  @param inError: error code
     */
    /* todo define enum UspiErrorCodes as input parameter type */
    virtual void onTouchError(int inError) { (void)inError; }

    /** @brief callback initiated when screen resolution is set later and margins are necessary
     *  @param inWidth: new resolution width
     *  @param inHeight: new resolution height
     *  @param inMarginX: margin value on the x-axis
     *  @param inMarginY: margin value on the y-axis
     * */
    virtual void onLateResolutionData(int inWidth, int inHeight, int inMarginX, int inMarginY)
    {
        (void)inWidth;
        (void)inHeight;
        (void)inMarginX;
        (void)inMarginY;
    }

    /** @brief callback initiated when USPI wants to log
     *  @param inLogLevel: log level ranging from verbose(5) to fatal(0)
     *  @param inLogString: Log string to be printed
     */
    virtual void onLogging(UspiLogLevel inLogLevel, const std::string& inLogString) = 0;
};

/** @interface  ITouchFacade
 *  @brief      Facade to simplify the underlying touch implementation.
 */
class ITouchFacade
{
public:
    ITouchFacade(void* inSessionCtx) { (void)inSessionCtx; };

    virtual ~ITouchFacade() { };

    /** @brief  initialize the facade once when starting
     *  @param  inCallbacks callbacks to notify the upper layer
     *  @return true on success, false on failure
     */
    virtual bool initialize(ITouchFacadeCallbacks* inCallbacks) = 0;
    /** @brief  deinitialize the facade once when shutting down
     *  @return true on success, false on failure
     */
    virtual bool deinitialize() = 0;

    /** @brief  connect to touch whenever head unit is on projected mode and
     *          SPI component gets the screen
     *  @return true on success, false on failure
     */
    virtual bool connect() = 0;
    /** @brief  disconnect to touch whenever head unit is on native mode and
     *          SPI component loses the screen
     *  @return true on success, false on failure
     */
    virtual bool disconnect() = 0;
};

} } /* namespace adit { namespace uspi { */

#endif /* USPI_I_TOUCH_FACADE_H */
